//
//  Hall.swift
//
//  Copyright © 2020 Apple Inc. All rights reserved.
//

import SpriteKit
import SPCScene
import Canvas
import SwansQuest


public class HallEntrance : Scene, SceneAnimating {
    static let backgrounds = [#imageLiteral(resourceName: "Tunnel-1024-0.png"), #imageLiteral(resourceName: "Tunnel-1024-1.png"), #imageLiteral(resourceName: "Tunnel-1024-2.png"), #imageLiteral(resourceName: "Tunnel-1024-3.png")]
    static let ground = -384.0
    
    var hero : Hero
    var bg: Graphic
    
    public override init() {
        hero = Hero()
        bg = Graphic(image: Image(with: HallEntrance.backgrounds[0]), name: "Entrance to Swan Castle")
        
        super.init(size: Scene.sceneSize)
        
        self.place(bg, at: Point.center)
        self.place(hero, at: Point(x: -448.0, y: HallEntrance.ground))
    }
    
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    // MARK: - SceneAnimating
    
    public var sceneTransition: SKTransition?
    public var animationDelegate: SceneAnimatingDelegate?
    public var loadingDescription: String? {
        didSet {
            enableAccessibility(label: loadingDescription!)
        }
    }
    
    public func sceneIsPresented() {
        let backgrounds = HallEntrance.backgrounds.map { Image(with: $0) }
        bg.animate(with: backgrounds, fps: 4, loop: true)
        animationDelegate?.sceneDidLoad(self)
        
        hero.approachHall {
            self.animationDelegate?.sceneDidFinishAnimations(self)
        }
    }
    
    public func sceneWillDismiss() {
        
    }
}

public class Hall : Scene, SceneAnimating {
    static let backgrounds = [#imageLiteral(resourceName: "QueensHall_1024-0.png"), #imageLiteral(resourceName: "QueensHall_1024-1.png")]
    static let ground = -288.0
    static let throne = -144.0
    
    var hero: Hero
    var swan: Swan
    var bg: Graphic
    
    public enum Mode {
        case coming, comingWithScroll, comingWithChest
        case leaving, leavingWithChest
        
        var distance: Double {
            switch self {
            case .coming, .comingWithScroll, .comingWithChest:
                return 300.0
            default:
                return -300.00
            }
        }
        
        func startAnimation(on hero: Hero, completion: @escaping (()-> Void)) {
            hero.walk(distance: self.distance, completion: {})
            hero.run(SKAction.wait(forDuration: 4.0)) {
                completion()
            }
        }
        
        func setup(character: Hero, in scene: Scene) {
            switch self {
            case .coming:
                scene.place(character, at: Point(x: -448.0, y: Hall.ground))
            case .comingWithScroll:
                scene.place(character, at: Point(x: -448.0, y: Hall.ground))
                character.holding = .scroll
            case .comingWithChest:
                scene.place(character, at: Point(x: -448.0, y: Hall.ground))
                character.holding = .scrollChest
            case .leaving:
                scene.place(character, at: Point(x: 0, y: Hall.ground))
                character.holding = .scroll
                character.turn()
            case .leavingWithChest:
                scene.place(character, at: Point(x: 0, y: Hall.ground))
                character.holding = .scrollChest
                character.turn()
            }
        }
    }
    var mode: Mode
    
    public init(heroIs mod: Mode) {
        hero = Hero()
        swan = Swan()
        bg = Graphic(image: Image(with: Hall.backgrounds[0]), name: "Swan Castle")
        mode = mod
        
        super.init(size: Scene.sceneSize)
        
        self.place(bg, at: Point.center)
        self.place(swan, at: Point(x: 384.0, y: Hall.throne))
        
        mode.setup(character: hero, in: self)
        swan.idle()
    }
    
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    // MARK: - SceneAnimating
    
    public var sceneTransition: SKTransition?
    public var animationDelegate: SceneAnimatingDelegate?
    public var loadingDescription: String? {
        didSet {
            enableAccessibility(label: loadingDescription!)
        }
    }
    
    public func sceneIsPresented() {
        let backgrounds = Hall.backgrounds.map { Image(with: $0) }
        bg.animate(with: backgrounds, fps: 4/3, loop: true)
        animationDelegate?.sceneDidLoad(self)
        
        mode.startAnimation(on: hero) {
            self.animationDelegate?.sceneDidFinishAnimations(self)
        }
    }
    
    public func sceneWillDismiss() {
        
    }
}
